# coding: utf-8
from sqlalchemy import Column, DateTime, Index, String, TIMESTAMP, Text, text, FetchedValue
from sqlalchemy.dialects.mysql import INTEGER, TINYINT, VARCHAR
from werkzeug.security import generate_password_hash, check_password_hash

from models.base_model import Base


class Brand(Base):
    __tablename__ = 'brand'

    id = Column(INTEGER(10), primary_key=True)
    brand_name = Column(VARCHAR(191), nullable=False, index=True, comment='品牌名')
    logo = Column(VARCHAR(191), nullable=False, comment='logo')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class Cate(Base):
    __tablename__ = 'cate'

    id = Column(INTEGER(10), primary_key=True)
    cate_name = Column(VARCHAR(191), nullable=False, index=True, comment='分类名')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class Hatch(Base):
    __tablename__ = 'hatch'
    __table_args__ = (
        Index('hatch_machine_UNIQUE', 'machine_no', 'hatch_no', unique=True),
    )

    id = Column(INTEGER(10), primary_key=True, unique=True)
    machine_no = Column(String(20, 'utf8mb4_unicode_ci'), nullable=False, comment='机柜id')
    hatch_no = Column(TINYINT(3), nullable=False, comment='机柜仓口号')
    production_id = Column(INTEGER(10), nullable=False, comment='商品id')
    name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, comment='商品名称')
    title = Column(String(200, 'utf8mb4_unicode_ci'), nullable=False, comment='商品标题')
    brand_id = Column(INTEGER(10), nullable=False, comment='品牌ID')
    brand_name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, comment='商品名称')
    cate_id = Column(INTEGER(10), nullable=False, comment='分类ID')
    cate_name = Column(String(200, 'utf8mb4_unicode_ci'), nullable=False, comment='商品标题')
    price = Column(INTEGER(10), nullable=False, comment='价格')
    original_price = Column(INTEGER(10), nullable=False, comment='商品原价')
    img = Column(String(200, 'utf8mb4_unicode_ci'))
    tags = Column(String(255, 'utf8mb4_unicode_ci'), comment='商品标签')
    content = Column(Text(collation='utf8mb4_unicode_ci'), comment='商品内容')
    summary = Column(Text(collation='utf8mb4_unicode_ci'), comment='商品描述')
    status = Column(TINYINT(3), nullable=False, server_default=text("'1'"),
                    comment='1可售2售空')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class Machine(Base):
    __tablename__ = 'machine'

    id = Column(INTEGER(10), primary_key=True)
    machine_no = Column(String(20, 'utf8mb4_unicode_ci'), nullable=False, unique=True, comment='机柜编号')
    qrcode_no = Column(String(20, 'utf8mb4_unicode_ci'), nullable=False, unique=True, comment='二维码编号')
    mac = Column(String(30, 'utf8mb4_unicode_ci'), nullable=False, unique=True, comment='mac')
    power = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='电量')
    short_address = Column(VARCHAR(45))
    address = Column(String(191, 'utf8mb4_unicode_ci'), comment='机柜位置')
    place_id = Column(INTEGER(10), nullable=False)
    mch_platform = Column(INTEGER(11), nullable=False, server_default=text("'1'"), comment='1咻咻')
    position = Column(String(20, 'utf8mb4_unicode_ci'), comment='机柜位置坐标')
    hatch_number = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='机柜的仓口数量')
    type = Column(TINYINT(3), nullable=False, server_default=text("'1'"), comment='机柜类型1正常')
    status = Column(TINYINT(1), server_default=text("'1'"), comment='状态: 1正常-1删除')
    created_at = Column(TIMESTAMP, nullable=False, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, nullable=False, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class MachineProduction(Base):
    __tablename__ = 'machine_production'

    id = Column(INTEGER(10), primary_key=True)
    machine_id = Column(INTEGER(10), nullable=False)
    production_id = Column(INTEGER(10), nullable=False)
    hatch_no = Column(INTEGER(10), nullable=False)
    status = Column(TINYINT(1), nullable=False, comment='状态: 1整除-1删除')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class NfcCard(Base):
    __tablename__ = 'nfc_card'
    __table_args__ = {'comment': 'nfc卡片表'}

    id = Column(INTEGER(10), primary_key=True)
    card_no = Column(String(40, 'utf8mb4_unicode_ci'), comment='卡片编号')
    user_id = Column(INTEGER(10), nullable=False, index=True, comment='用户id')
    nick_name = Column(String(40, 'utf8mb4_unicode_ci'), comment='学生名称')
    phone = Column(String(40, 'utf8mb4_unicode_ci'), index=True, comment='手机号')
    money = Column(INTEGER(10), nullable=False)
    mch_platform = Column(INTEGER(11), nullable=False, server_default=text("'1'"), comment='1咻咻')
    status = Column(TINYINT(4), nullable=False, comment='状态0停用1正常')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class NfcCardPayRecord(Base):
    __tablename__ = 'nfc_card_pay_record'
    __table_args__ = {'comment': 'nfc卡片充值表'}

    id = Column(INTEGER(10), primary_key=True)
    card_no = Column(String(40, 'utf8mb4_unicode_ci'), index=True, comment='卡片编号')
    rent_no = Column(VARCHAR(40), nullable=False, comment='租借单号')
    user_id = Column(INTEGER(10), nullable=False, index=True, comment='用户id')
    is_pay = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='是否支付')
    pay_money = Column(INTEGER(10), nullable=False, comment='充值金额')
    mch_platform = Column(INTEGER(11), nullable=False, server_default=text("'1'"), comment='1咻咻')
    prepay_id = Column(VARCHAR(191), comment='微信支付prepay_id')
    refund_no = Column(VARCHAR(191), comment='退款单号')
    status = Column(TINYINT(4), nullable=False, comment='状态-1停用1正常')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class NfcCardPayRefund(Base):
    __tablename__ = 'nfc_card_pay_refund'

    id = Column(INTEGER(10), primary_key=True)
    refund_no = Column(String(191, 'utf8mb4_bin'), nullable=False)
    rent_no = Column(String(191, 'utf8mb4_bin'), nullable=False)
    fee = Column(INTEGER(10), server_default=FetchedValue(), nullable=False)
    comment = Column(Text(collation='utf8mb4_bin'))
    cause = Column(String(191, 'utf8mb4_bin'))


class Place(Base):
    __tablename__ = 'place'

    id = Column(INTEGER(10), primary_key=True)
    place_name = Column(VARCHAR(191), nullable=False, index=True, comment='场所名')
    img = Column(VARCHAR(191), comment='展示界面的图片')
    logo = Column(VARCHAR(191), nullable=False, comment='微型头像')
    address = Column(VARCHAR(255), nullable=False, server_default=text("''"))
    position = Column(String(20, 'utf8mb4_unicode_ci'))
    open_time = Column(VARCHAR(191), nullable=False, comment='开始时间')
    close_time = Column(VARCHAR(191), nullable=False, comment='结束时间')
    open_week = Column(VARCHAR(255), nullable=False, server_default=text("''"))
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class Production(Base):
    __tablename__ = 'production'

    id = Column(INTEGER(10), primary_key=True)
    name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, index=True, comment='商品名称')
    title = Column(String(200, 'utf8mb4_unicode_ci'), nullable=False, comment='商品标题')
    brand_id = Column(INTEGER(10), nullable=False, comment='品牌ID')
    cate_id = Column(INTEGER(10), nullable=False, comment='分类ID')
    price = Column(INTEGER(10), nullable=False, comment='价格')
    original_price = Column(INTEGER(10), nullable=False, comment='商品原价')
    img = Column(String(200, 'utf8mb4_unicode_ci'))
    tags = Column(String(255, 'utf8mb4_unicode_ci'), comment='商品标签')
    content = Column(Text(collation='utf8mb4_unicode_ci'), comment='商品内容')
    summary = Column(Text(collation='utf8mb4_unicode_ci'), comment='商品描述')
    status = Column(TINYINT(1), comment='状态: 1正常-1删除')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class Rent(Base):
    __tablename__ = 'rent'

    id = Column(INTEGER(10), primary_key=True)
    rent_no = Column(VARCHAR(40), nullable=False, index=True, comment='租借单号')
    machine_no = Column(String(40, 'utf8mb4_unicode_ci'), nullable=False, index=True, comment='机柜id')
    user_id = Column(INTEGER(10), nullable=False, index=True, comment='用户id')
    place_id = Column(INTEGER(10), nullable=False, index=True, comment='场所id')
    total = Column(INTEGER(10), server_default=text("'0'"), comment='应收金额')
    real_total = Column(INTEGER(10), server_default=text("'0'"), comment='实收金额')
    agent_total = Column(INTEGER(10), server_default=text("'0'"), comment='给代理商看的收入')
    back_money = Column(INTEGER(10), nullable=False, server_default=text("'0'"), comment='退款金额')
    is_pay = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='是否支付')
    rent_type = Column(TINYINT(3), nullable=False, server_default=text("'1'"), comment='租借类型1现场租借2预约3nfc租借')
    mch_platform = Column(INTEGER(1), nullable=False, server_default=text("'1'"), comment='1待定')
    add_time = Column(TIMESTAMP, nullable=False, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"),
                      comment='下单时间')
    pay_time = Column(TIMESTAMP, comment='支付时间')
    is_over = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='是否完结')
    is_cancel = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='是否取消交易')
    refund_no = Column(VARCHAR(191), comment='退款单号')
    expire_handle = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='是否做过期处理')
    prepay_id = Column(VARCHAR(191), comment='微信支付prepay_id')
    over_time = Column(TIMESTAMP, comment='订单完结时间')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class RentDetail(Base):
    __tablename__ = 'rent_detail'

    id = Column(INTEGER(10), primary_key=True, unique=True)
    rent_no = Column(String(40, 'utf8mb4_unicode_ci'), nullable=False, comment='订单编号')
    user_id = Column(INTEGER(10), nullable=False, index=True, comment='用户id')
    machine_no = Column(String(20, 'utf8mb4_unicode_ci'), nullable=False, comment='机柜id')
    hatch_no = Column(TINYINT(3), nullable=False, comment='机柜仓口号')
    production_id = Column(INTEGER(10), nullable=False, comment='商品id')
    is_take = Column(TINYINT(3), nullable=False, server_default=text("'0'"), comment='是否取货')
    name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, comment='商品名称')
    title = Column(String(200, 'utf8mb4_unicode_ci'), nullable=False, comment='商品标题')
    brand_id = Column(INTEGER(10), nullable=False, comment='品牌ID')
    brand_name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, comment='商品名称')
    cate_id = Column(INTEGER(10), nullable=False, comment='分类ID')
    cate_name = Column(String(200, 'utf8mb4_unicode_ci'), nullable=False, comment='商品标题')
    price = Column(INTEGER(10), nullable=False, comment='价格')
    img = Column(String(200, 'utf8mb4_unicode_ci'))
    tags = Column(String(255, 'utf8mb4_unicode_ci'), comment='商品标签')
    content = Column(Text(collation='utf8mb4_unicode_ci'), comment='商品内容')
    summary = Column(Text(collation='utf8mb4_unicode_ci'), comment='商品描述')
    status = Column(TINYINT(3), nullable=False, server_default=text("'1'"), comment='1正常  -1删除')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class SalePlan(Base):
    __tablename__ = 'sale_plan'

    id = Column(INTEGER(10), primary_key=True)
    name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, comment='方案名称')
    title = Column(String(200, 'utf8mb4_unicode_ci'), nullable=False, comment='方案标题')
    status = Column(TINYINT(1), nullable=False, comment='状态: 1整除-1删除')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class SalePlanMachine(Base):
    __tablename__ = 'sale_plan_machine'

    id = Column(INTEGER(10), primary_key=True)
    plan_id = Column(INTEGER(10), nullable=False)
    machine_id = Column(INTEGER(10), nullable=False)
    status = Column(TINYINT(1), nullable=False, comment='状态: 1正常 -1删除')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class SalePlanProduction(Base):
    __tablename__ = 'sale_plan_production'

    id = Column(INTEGER(10), primary_key=True)
    plan_id = Column(INTEGER(10), nullable=False)
    production_id = Column(INTEGER(10), nullable=False)
    index = Column(INTEGER(10), nullable=False)
    status = Column(TINYINT(1), nullable=False, comment='状态: 1整除-1删除')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class TallymanAccount(Base):
    __tablename__ = 'tallyman_account'

    id = Column(INTEGER(10), primary_key=True, unique=True)
    user_no = Column(String(25, 'utf8mb4_unicode_ci'), nullable=False, unique=True)
    user_name = Column(String(255, 'utf8mb4_unicode_ci'), nullable=False)
    phone = Column(String(191, 'utf8mb4_unicode_ci'), nullable=False, unique=True)
    level = Column(INTEGER(1), nullable=False, comment='1:补货员')
    status = Column(INTEGER(1), nullable=False, comment='1:正常 2：删除')
    _password_hash_ = Column(String(255, 'utf8mb4_unicode_ci'))
    comment = Column(String(255, 'utf8mb4_unicode_ci'))
    last_login = Column(DateTime)
    expire_time = Column(DateTime)
    created_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))

    @property
    def password(self):
        raise Exception('密码不能被读取')  # 为了保持使用习惯，还是设置一个password字段用来设置密码，当然也不能被读取。

    # 赋值password，则自动加密存储。
    @password.setter
    def password(self, value):
        self._password_hash_ = generate_password_hash(value)

    # 使用check_password,进行密码校验，返回True False。
    def check_password(self, pasword):
        return check_password_hash(self._password_hash_, pasword)


class TallymanMachine(Base):
    __tablename__ = 'tallyman_machine'
    __table_args__ = (
        Index('index4tallymachine_user_machine_unique', 'user_id', 'machine_no', unique=True),
    )

    id = Column(INTEGER(10), primary_key=True)
    user_id = Column(INTEGER(10), nullable=False, index=True)
    machine_no = Column(String(20, 'utf8mb4_unicode_ci'), nullable=False, comment='机柜id')
    status = Column(INTEGER(1), nullable=False, index=True, comment='1:正常 -1：删除')
    created_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class TallymanLoginRecord(Base):
    __tablename__ = 'tallyman_login_record'

    id = Column(INTEGER, primary_key=True)
    phone = Column(String(255, 'utf8mb4_bin'), nullable=False, server_default=FetchedValue())
    ip = Column(String(255, 'utf8mb4_bin'), nullable=False, server_default=FetchedValue())
    platform = Column(INTEGER, nullable=False, server_default=FetchedValue(), comment='平台 2小导游 8商户PC')
    last_login = Column(DateTime, nullable=False, server_default=FetchedValue())
    login_type = Column(INTEGER, nullable=False, server_default=FetchedValue(), comment='1:验证码登录 2：token 3:发送验证码  4:密码')


class TallyRecord(Base):
    __tablename__ = 'tally_record'
    __table_args__ = (
        Index('hatch_machine_UNIQUE', 'machine_no', 'hatch_no', unique=True),
    )

    id = Column(INTEGER(10), primary_key=True, unique=True)
    user_no = Column(String(25, 'utf8mb4_unicode_ci'), nullable=False, unique=True)
    user_name = Column(String(255, 'utf8mb4_unicode_ci'), nullable=False)
    machine_no = Column(String(20, 'utf8mb4_unicode_ci'), nullable=False, comment='机柜id')
    hatch_no = Column(TINYINT(3), nullable=False, comment='机柜仓口号')
    production_id = Column(INTEGER(10), nullable=False, comment='商品id')
    production_name = Column(String(100, 'utf8mb4_unicode_ci'), nullable=False, comment='商品名称')
    status = Column(TINYINT(3), nullable=False, server_default=text("'1'"), comment='1指令已下发(等待开仓) 2指令上报(补货完成)')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))


class WxUser(Base):
    __tablename__ = 'wx_user'
    __table_args__ = {'comment': '微信用户表'}

    id = Column(INTEGER(10), primary_key=True)
    openid = Column(String(40, 'utf8mb4_unicode_ci'), index=True, comment='微信支付宝公众平台openID')
    unionid = Column(String(40, 'utf8mb4_unicode_ci'), comment='微信支付宝unionid')
    platform = Column(TINYINT(4), nullable=False, server_default=text("'0'"), comment='平台')
    phone = Column(String(40, 'utf8mb4_unicode_ci'), index=True, comment='手机号')
    language = Column(String(40, 'utf8mb4_unicode_ci'), comment='语种')
    nick_name = Column(String(40, 'utf8mb4_unicode_ci'), comment='昵称')
    gender = Column(TINYINT(4), nullable=False, server_default=text("'0'"), comment='性别 0：未知、1：男、2：女')
    avatar_url = Column(String(191, 'utf8mb4_unicode_ci'), comment='头像')
    city = Column(String(45, 'utf8mb4_unicode_ci'))
    province = Column(String(45, 'utf8mb4_unicode_ci'))
    country = Column(String(45, 'utf8mb4_unicode_ci'))
    status = Column(TINYINT(4), nullable=False, comment='状态0停用1正常')
    last_login_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"),
                           comment='上次登录时间')
    created_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP"))
    updated_at = Column(TIMESTAMP, server_default=text("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"))
