#!usr/bin/env python
# -*- coding:utf-8 _*-
"""
@version:
author:Aeolus
@file: nfc_card_portal.py
"""
import json
import logging
import re
import time

from flask import Blueprint, request, jsonify, g
from sqlalchemy import extract
from sqlalchemy.exc import SQLAlchemyError

from config.base_config import NFC_PAY_LOAD_SECRET
from config.commen_config import USER_RENT_PREPAY_ID
from config.wechat_config import platform_appid_config_list, pay_config_list, NFC_PAY_CALLBCK_URL
from models.base_model import db
from models.models import NfcCard, NfcCardPayRecord, NfcCardPayRefund
from service.rent_service import RentService
from service.wechat_service import WeChatPayService
from utils.error_code import NFC_CARD_NOT_EXIST, NFC_CARD_ACTIVATED_ERROR, WE_MINIAPP_PAY_FAIL, NO_RENT_RECORD, \
    NO_NFC_CARD_ERROR, NFC_PAY_LOAD_SECRET_ERROR
from utils.my_redis_cache import redis_client
from utils.my_response import BaseResponse

logger = logging.getLogger(__name__)

nfc_card_route = Blueprint('nfc_card', __name__)


@nfc_card_route.route("activate", methods=["POST"])
def run_nfc_card_activate():
    json_data = request.get_json()
    card_no = json_data["card_no"]
    user_name = json_data["user_name"]
    phone = json_data.get("phone", None)

    card = NfcCard.query.filter_by(card_no=card_no).first()
    if not card:
        return jsonify(NFC_CARD_NOT_EXIST)
    if card.status != 0:
        return jsonify(NFC_CARD_ACTIVATED_ERROR)

    card.status = 1
    card.user_id = g.user.id
    if user_name:
        card.nick_name = user_name
    if phone:
        card.phone = phone
    db.session.add(card)
    db.session.commit()

    return BaseResponse()


@nfc_card_route.route("edit", methods=["POST"])
def run_nfc_card_edit():
    json_data = request.get_json()
    card_no = json_data["card_no"]
    user_name = json_data.get("user_name", None)
    phone = json_data.get("phone", None)
    limit_count = json_data.get("limit_count", None)

    card = NfcCard.query.filter_by(card_no=card_no, status=1).first()
    if not card:
        return jsonify(NFC_CARD_NOT_EXIST)

    if user_name:
        card.nick_name = user_name
    if phone:
        card.phone = phone
    if limit_count:
        card.limit_count = limit_count

    db.session.add(card)
    db.session.commit()

    return BaseResponse()


@nfc_card_route.route("delete", methods=["POST"])
def run_nfc_card_delete():
    json_data = request.get_json()
    card_no = json_data["card_no"]

    card = NfcCard.query.filter_by(card_no=card_no, status=1).first()
    if not card:
        return jsonify(NFC_CARD_NOT_EXIST)

    card.status = -1

    db.session.add(card)
    db.session.commit()

    return BaseResponse()


@nfc_card_route.route("my_cards", methods=["POST"])
def get_my_cards():
    json_data = request.get_json()
    page = json_data["page"]
    page_size = json_data["pageSize"]

    filter_list = [
        NfcCard.user_id == g.user.id,
        NfcCard.status.in_([1])
    ]
    total_count = NfcCard.query.filter(*filter_list).count()
    if not total_count:
        return BaseResponse(data=[], total=0, page=page, pageSize=page_size)
    cards = NfcCard.query.filter(*filter_list).offset((page - 1) * page_size).limit(page_size).all()
    if not cards:
        return jsonify(NO_NFC_CARD_ERROR)

    return_data = []
    for card in cards:
        tmp_data = {
            "card_no": card.card_no,
            "nick_name": card.nick_name,
            "phone": card.phone,
            "money": card.money,
            "status": card.status,
        }
        return_data.append(tmp_data)

    return BaseResponse(data=return_data, total=total_count, page=page, pageSize=page_size)


@nfc_card_route.route("pay", methods=["POST"])
def run_nfc_card_pay():
    json_data = request.get_json()
    card_no = json_data["card_no"]
    money = json_data["money"]
    user = g.user

    # 验证机柜是否存在
    card = NfcCard.query.filter_by(card_no=card_no, status=1).first()
    if not card:
        return jsonify(NFC_CARD_NOT_EXIST)

    # 生成订单编号
    rent_no = RentService.create_order_no()

    # 配置微信订单数据
    wechat_service = WeChatPayService(app_id=platform_appid_config_list[user.platform],
                                      config_name=pay_config_list[card.mch_platform])
    pay_data = {
        'body': '灰兔智能租借押金',  # 商品描述
        'out_trade_no': rent_no,  # 商户订单号
        'total_fee': money,  # 总价
        'trade_type': "JSAPI",
        'openid': user.openid,
        'timeStamp': str(int(time.time())),
        'attach': {
            "card_no": card_no,
            "user_id": user.id,
            "platform": card.mch_platform,
        }
    }

    # 微信下单接口
    pay_info = wechat_service.unifiedorder(pay_data, callback_url=NFC_PAY_CALLBCK_URL)
    if not pay_info:
        return BaseResponse(**WE_MINIAPP_PAY_FAIL)

    # 微信下单成功，redis存prepay_id 5分钟失效
    redis_client.set(USER_RENT_PREPAY_ID + str(user.id) + rent_no, pay_info["prepay_id"], 300)
    return BaseResponse(data=pay_info)


@nfc_card_route.route('/wx_pay_callback', methods=['GET', 'POST'])
def run_nfc_card_wx_pay_callback():
    """
    支付回调接口
    :return:
    """
    response_data = {
        'return_code': 'SUCCESS',
        'return_msg': 'SUCCESS'
    }
    error_data = {
        'return_code': 'FAIL',
        'return_msg': 'FAIL'
    }
    header = {'Content-Type': 'application/xml'}

    # 进行签名认证
    xml_data = request.data

    import xmltodict
    callback_data = dict(xmltodict.parse(xml_data)['xml'])
    logger.info(xml_data)
    logger.info(callback_data)
    attach = re.sub('\'', '\"', callback_data["attach"])
    rent_data = json.loads(attach)
    platform = rent_data["platform"]
    card_no = rent_data["card_no"]
    user_id = rent_data["user_id"]
    platform = pay_config_list[platform]
    wechat_service = WeChatPayService(app_id=callback_data["appid"], config_name=platform)
    verify_result = wechat_service.verify_pay_callbak_sign(callback_data)
    if not verify_result:
        return xmltodict.unparse({'xml': response_data}, pretty=True), header

    # 判断订单是否已经存在过
    rent_no = callback_data['out_trade_no']
    rent = NfcCardPayRecord.query.filter_by(rent_no=rent_no).first()
    if rent:
        return xmltodict.unparse({'xml': response_data}, pretty=True), header

    try:
        rent = NfcCardPayRecord()
        rent.rent_no = rent_no
        rent.card_no = card_no
        rent.user_id = user_id
        rent.pay_money = int(callback_data["total_fee"])
        rent.is_pay = 1
        rent.status = 1
        rent.mch_platform = platform
        prepay_id = redis_client.get(USER_RENT_PREPAY_ID + str(user_id) + rent_no)
        if prepay_id:
            rent.prepay_id = prepay_id

        db.session.add(rent)
        db.session.commit()

    except Exception as e:
        return xmltodict.unparse({'xml': error_data}, pretty=True), header

    return xmltodict.unparse({'xml': response_data}, pretty=True), header


@nfc_card_route.route('/pay_result', methods=['POST'])
def run_nfc_card_pay_result():
    json_data = request.get_json()
    rent_no = json_data["rent_no"]

    rent = NfcCardPayRecord.query.filter_by(rent_no=rent_no).first()
    if not rent:
        return jsonify(NO_RENT_RECORD)
    tmp_data = {}
    tmp_data["rent_no"] = rent.rent_no
    tmp_data["card_no"] = rent.card_no
    tmp_data["user_id"] = rent.user_id
    tmp_data["status"] = rent.status
    tmp_data["pay_money"] = rent.pay_money
    tmp_data["pay_date"] = rent.created_at.strftime("%Y-%m-%d %H:%M:%S")

    return BaseResponse(data=tmp_data)


@nfc_card_route.route('/pay_record', methods=['POST'])
def run_nfc_card_pay_record():
    json_data = request.get_json()
    page = json_data["page"]
    page_size = json_data["pageSize"]
    card_no = json_data.get("card_no", None)
    pay_year = json_data.get("pay_year", None)
    pay_month = json_data.get("pay_month", None)

    filter_list = [
        NfcCardPayRecord.user_id == g.user.id
    ]

    if card_no:
        filter_list.append(NfcCardPayRecord.card_no == card_no)
    if pay_year:
        filter_list.append(extract('year', NfcCardPayRecord.created_at) == pay_year)
    if pay_month:
        filter_list.append(extract('month', NfcCardPayRecord.created_at) == pay_month)

    order_list = [NfcCardPayRecord.created_at.desc()]

    total_count = NfcCardPayRecord.query.filter(*filter_list).count()
    if not total_count:
        return BaseResponse(data=[], total=total_count, page=page, pageSize=page_size)
    rent_list = NfcCardPayRecord.query.filter(*filter_list).order_by(*order_list).offset((page - 1) * page_size).limit(
        page_size).all()

    return_data = []
    for rent in rent_list:
        tmp_data = {}
        tmp_data["rent_no"] = rent.rent_no
        tmp_data["card_no"] = rent.card_no
        tmp_data["user_id"] = rent.user_id
        tmp_data["status"] = rent.status
        tmp_data["pay_money"] = rent.pay_money
        tmp_data["pay_date"] = rent.created_at.strftime("%Y-%m-%d %H:%M:%S")
        return_data.append(tmp_data)

    return BaseResponse(data=return_data, total=total_count, page=page, pageSize=page_size)


@nfc_card_route.route('/pay_refund', methods=['POST'])
def run_nfc_card_pay_refund():
    json_data = request.get_json()
    rent_no = json_data["rent_no"]
    comment = json_data.get("comment", None)
    cause = json_data.get("cause", None)

    rent = NfcCardPayRecord.query.filter_by(rent_no=rent_no, status=1).first()
    if not rent:
        return jsonify(NO_RENT_RECORD)

    # 退款操作
    data = {
        "out_refund_no": RentService.create_refund_no(),
        "out_trade_no": rent.rent_no,
        "total_fee": rent.pay_money,
        "refund_fee": rent.pay_money
    }
    result = WeChatPayService(app_id=platform_appid_config_list[g.user.platform],
                              config_name=pay_config_list[rent.mch_platform]).do_refund(data)

    if result:
        rent_refund = NfcCardPayRefund()
        rent_refund.refund_no = data["out_refund_no"]
        rent_refund.rent_no = rent_no
        rent_refund.fee = data["refund_fee"]
        rent_refund.comment = comment
        rent_refund.cause = cause

        rent.status = 3
        db.session.add(rent_refund)
        db.session.commit()

        return BaseResponse()


# 查询充值记录
@nfc_card_route.route('/user_pay_record', methods=['POST'])
def run_nfc_card_user_pay_record():
    json_data = request.get_json()
    secret = json_data["secret"]
    card_no = json_data["card_no"]
    page = json_data.get("page", 1)
    page_size = json_data.get("page_size", 5)

    if secret != NFC_PAY_LOAD_SECRET:
        return jsonify(NFC_PAY_LOAD_SECRET_ERROR)

    card_result = NfcCard.query.filter_by(card_no=card_no, status=1).first()
    if not card_result:
        return jsonify(NFC_CARD_NOT_EXIST)

    total_count = NfcCardPayRecord.query.filter_by(card_no=card_no, status=1).count()
    if not total_count:
        return BaseResponse(data=[], total_count=0, page=page, page_size=page_size)

    pay_records = NfcCardPayRecord.query.filter_by(card_no=card_no, status=1).all()
    result_data = []
    for record in pay_records:
        tmp_data = {
            "card_no": record.card_no,
            "record_no": record.rent_no,
            "pay_money": record.pay_money,
            "pay_time": record.created_at.strftime("%Y-%m-%d %H:%M:%S"),
        }
        result_data.append(tmp_data)

    return BaseResponse(data=result_data, total=total_count, page=page, page_size=page_size)


# 充值成功后改变数据库
@nfc_card_route.route('/load_succeed', methods=['POST'])
def run_nfc_card_load_succeed():
    json_data = request.get_json()
    secret = json_data["secret"]
    record_nos = json_data["record_nos"]
    card_no = json_data["card_no"]

    if secret != NFC_PAY_LOAD_SECRET:
        return jsonify(NFC_PAY_LOAD_SECRET_ERROR)

    card_result = NfcCard.query.filter_by(card_no=card_no).first()  # 查询到卡号
    if not card_result:
        return jsonify(NFC_CARD_NOT_EXIST)
    for record_no in record_nos:
        card_record = NfcCardPayRecord.query.filter_by(rent_no=record_no, status=1).first()  # 查询到充值记录
        if card_record:
            card_result.money += card_record.pay_money
            card_record.status = 2
            db.session.add(card_record)
    try:
        db.session.add(card_result)
        db.session.commit()
    except SQLAlchemyError as e:
        db.session.rollback()
    return_data = {
        "card_no": card_result.card_no,
        "money": card_result.money
    }
    return BaseResponse(data=return_data)


# 查询充值记录
@nfc_card_route.route('/user_load_record', methods=['POST'])
def run_nfc_card_user_load_record():
    json_data = request.get_json()
    secret = json_data["secret"]
    card_no = json_data["card_no"]
    page = json_data.get("page", 1)
    page_size = json_data.get("page_size", 5)

    if secret != NFC_PAY_LOAD_SECRET:
        return jsonify(NFC_PAY_LOAD_SECRET_ERROR)

    card_result = NfcCard.query.filter_by(card_no=card_no, status=1).first()
    if not card_result:
        return jsonify(NFC_CARD_NOT_EXIST)

    total_count = NfcCardPayRecord.query.filter_by(card_no=card_no, status=2).count()
    if not total_count:
        return BaseResponse(data=[], total_count=0, page=page, page_size=page_size)

    pay_records = NfcCardPayRecord.query.filter_by(card_no=card_no, status=2).all()
    result_data = []
    for record in pay_records:
        tmp_data = {
            "card_no": record.card_no,
            "record_no": record.rent_no,
            "load_money": record.pay_money,
            "load_time": record.updated_at.strftime("%Y-%m-%d %H:%M:%S"),
        }
        result_data.append(tmp_data)

    return BaseResponse(data=result_data, total=total_count, page=page, page_size=page_size)
