# -*- coding: utf-8 -*-
import datetime
import hashlib, uuid
from wechatpy import WeChatClient
from wechatpy.pay import WeChatPay, calculate_signature
import logging
from wechatpy.session.redisstorage import RedisStorage
from config.commen_config import WX_SESSION_KEY
from config.wechat_config import pay_config_dict, platform_config_dict
from utils.my_redis_cache import redis_client

logger = logging.getLogger(__name__)


class WeChatPayService(WeChatPay):
    """
     pass
    """

    def __init__(self, app_id, config_name):
        self.app_id = app_id
        self.config = pay_config_dict[config_name]
        mch_id = self.config["mch_id"]
        pay_key = self.config["pay_key"]
        cert_path = self.config["cert_path"]
        key_path = self.config["key_path"]
        super(WeChatPayService, self).__init__(app_id, pay_key, mch_id, mch_cert=cert_path, mch_key=key_path)

    def unifiedorder(self, pay_data):
        """

        :param pay_data:
        :return:
        """
        notify_url = self.config["domain"] + self.config["callback_url"]
        logger.debug(notify_url)
        pay_data["nonce_str"] = self.get_nonce_str()
        result = self.order.create(notify_url=notify_url, **pay_data)
        prepay_id = result.get('prepay_id')
        pay_sign_data = {
            'appId': self.app_id,
            'timeStamp': pay_data.get('timeStamp'),
            'nonceStr': pay_data.get('nonce_str'),
            'package': 'prepay_id={0}'.format(prepay_id),
            'signType': 'MD5',
        }
        pay_sign = calculate_signature(pay_sign_data, self.config["pay_key"])
        pay_sign_data.pop('appId')
        pay_sign_data['paySign'] = pay_sign
        pay_sign_data['prepay_id'] = prepay_id
        pay_sign_data['out_trade_no'] = pay_data.get('out_trade_no')
        # TODO  加上时间，金额，个数
        pay_sign_data['rent_time'] = datetime.datetime.fromtimestamp(int(pay_data.get('timeStamp'))).strftime(
            '%Y-%m-%d %H:%M:%S')
        pay_sign_data['total_fee'] = pay_data["total_fee"]
        pay_sign_data["number"] = pay_data["attach"]["number"]
        return pay_sign_data

    def do_refund(self, refund_data):
        """

        :param refund_data: 
        :return: 
        """
        for i in range(3):
            try:

                result = self.refund.apply(**refund_data)
                return_code = result['return_code']
                result_code = result.get('result_code')
                if return_code != 'SUCCESS' or result_code != 'SUCCESS':
                    continue
                else:
                    if result.get("err_code", None):
                        continue
                    return result
            except Exception as e:
                logger.info(e)
                continue

    @staticmethod
    def get_nonce_str():
        '''
        获取随机字符串
        :return:
        '''
        return str(uuid.uuid4()).replace('-', '')

    def create_sign(self, pay_data, pay_key):
        '''
        生成签名
        :return:
        '''

        stringA = '&'.join(["{0}={1}".format(k, pay_data.get(k)) for k in sorted(pay_data)])
        stringSignTemp = '{0}&key={1}'.format(stringA, pay_key)
        logger.info(stringSignTemp)
        sign = hashlib.md5(stringSignTemp.encode("utf-8")).hexdigest()
        return sign.upper()

    def verify_pay_callbak_sign(self, callback_data):
        """
        验证签名方法
        :param callback_data: 微信回调的xml解析后的数据
        :return:
        """

        if callback_data['result_code'] != 'SUCCESS':
            return False

        sign = callback_data['sign']
        callback_data.pop('sign')
        gene_sign = calculate_signature(callback_data, self.config["pay_key"])
        if sign != gene_sign:
            return False

        return True

    def query_refund_info(self, refund_data):
        result = self.refund.query(**refund_data)
        return result


class WxMPService(WeChatClient):

    def __init__(self, config_name):
        config = platform_config_dict[config_name]
        app_id = config["app_id"]
        app_secret = config["app_secret"]

        session_interface = RedisStorage(
            redis_client,
            prefix="wx"
        )
        super(WxMPService, self).__init__(app_id, app_secret, session=session_interface)

    def code_to_session(self, code):

        res = self.wxa.code_to_session(code)
        open_id = res["openid"]
        session_key = res["session_key"]
        try:
            old_session_key = redis_client.get(WX_SESSION_KEY + open_id)
        except:
            old_session_key = None
        if old_session_key:
            res["session_key"] = str(old_session_key, encoding='utf-8')
        try:
            redis_client.set(WX_SESSION_KEY + open_id, session_key)
        except Exception as e:
            raise e
        return res


if __name__ == '__main__':
    target_wechat = WeChatPayService()
