# -*- coding: utf-8 -*-
import datetime
import logging
import time

from flask import Blueprint, jsonify, request, g
from config.base_config import MONGO_DATABASE_URI
from config.commen_config import LOGIN_TYPE

from models.base_model import db
from models.models import TallymanAccount, TallymanMachine, TallymanLoginRecord, Machine, Hatch, TallyRecord
from service.tallyman_service import TallymanService
from utils.error_code import TALLYMAN_ACCOUNT_EXIST, PHONE_NOT_VALID_ERROR, TOKEN_NOT_VALID_ERROR, PASSWORD_ERROR, \
    TALLYMAN_ACCOUNT_NOT_EXIST, MACHINE_NOT_EXIST_ERROR, HATCH_NOT_EXIST_ERROR
from utils.jwt_util import verify_jwt, generate_jwt
from utils.my_response import BaseResponse

logger = logging.getLogger(__name__)

tallyman_route = Blueprint('tallyman', __name__)


@tallyman_route.route('/test')
def test():
    ip = request.remote_addr
    return BaseResponse(**{'code': 200, 'msg': 'success!', 'data': ip})


@tallyman_route.route('/edit_password', methods=['GET', 'POST'])
def run_tallyman_edit_password():
    json_data = request.get_json()
    password = json_data.get('password', None)

    tallyman_info = g.user
    if not tallyman_info:
        return jsonify(TALLYMAN_ACCOUNT_NOT_EXIST)

    if password:
        tallyman_info.password = password

    db.session.add(tallyman_info)
    db.session.commit()
    return BaseResponse()


@tallyman_route.route('/edit_account', methods=['GET', 'POST'])
def run_tallyman_edit_account():
    admin = g.user
    if g.user.level != 1:
        return jsonify({"error_code": "500", "error_message": "没有权限"})
    json_data = request.get_json()
    phone = json_data.get('phone', None)
    user_name = json_data.get('name', None)
    password = json_data.get('password', None)
    machine_list = json_data.get('machine_list', None)

    tallyman_info = TallymanAccount.query.filter_by(phone=phone).first()
    if not tallyman_info:
        return jsonify(TALLYMAN_ACCOUNT_NOT_EXIST)

    if user_name:
        tallyman_info.user_name = user_name

    if password:
        tallyman_info.password = password

    db.session.add(tallyman_info)
    db.session.commit()

    if not machine_list:
        return BaseResponse()

    old_machine_list = Machine.query.join(TallymanMachine,
                                          TallymanMachine.machine_no == Machine.machine_no).filter(
        TallymanMachine.user_id == tallyman_info.id).all()
    old_machine_dict = {}
    for i in old_machine_list:
        old_machine_dict[i.machine_no] = i
    for i in machine_list:
        if old_machine_dict.get(i):
            old_machine_dict[i].status = 1
            db.session.add(old_machine_dict[i])
        else:
            model = TallymanMachine()
            model.user_id = tallyman_info.id
            model.machine_no = i
            model.status = 1
            db.session.add(model)
    db.session.commit()
    return BaseResponse()


@tallyman_route.route('/add_account', methods=['GET', 'POST'])
def add_user():
    admin = g.user
    if g.user.level != 1:
        return jsonify({"error_code": "500", "error_message": "没有权限"})
    json_data = request.get_json()
    user_name = json_data['name'] if 'name' in json_data else 'SSW'
    phone = json_data['phone'] if 'phone' in json_data else None
    level = int(json_data['level']) if 'level' in json_data else 2
    password = json_data['password'] if 'password' in json_data else None
    comment = json_data['comment'] if 'comment' in json_data else ''
    machine_list = json_data.get("machine_list", [])

    tallyman = TallymanAccount.query.filter_by(phone=phone).first()
    if tallyman:
        if tallyman.status != -1:
            return jsonify(TALLYMAN_ACCOUNT_EXIST)

    else:
        tallyman = TallymanAccount()

    tallyman.user_no = "todo"
    tallyman.user_name = user_name
    tallyman.phone = phone
    tallyman.level = level
    tallyman.status = 1
    tallyman.comment = comment
    tallyman.created_at = datetime.datetime.now()
    tallyman.updated_at = datetime.datetime.now()
    if password:
        tallyman.password = password
    db.session.add(tallyman)
    db.session.commit()
    tallyman.user_no = "SK" + str(tallyman.id).zfill(6)
    db.session.add(tallyman)
    db.session.commit()

    if not machine_list:
        return BaseResponse()

    for i in machine_list:
        model = TallymanMachine()
        model.user_id = tallyman.id
        model.machine_no = i
        model.status = 1
        db.session.add(model)

    db.session.commit()
    return BaseResponse()


@tallyman_route.route('/delete_account', methods=['GET', 'POST'])
def delete_user():
    json_data = request.get_json()
    phone = json_data['phone']

    tallyman = TallymanAccount.query.filter_by(phone=phone).first()
    if not tallyman:
        return BaseResponse()
    tallyman.status = -1
    db.session.add(tallyman)

    agent_spot_info = TallymanMachine.query.filter_by(user_id=tallyman.id).all()
    for info in agent_spot_info:
        info.status = -1
        db.session.add(info)

    db.session.commit()

    return BaseResponse()


@tallyman_route.route('/login', methods=['GET', 'POST'])
def run_tallyman_login():
    cur_ip = request.remote_addr
    json_data = request.get_json()

    data = {}
    phone = json_data['phone']
    password = json_data['password']

    # 密码登录
    # 判断密码是否正确
    tallyman = TallymanAccount.query.filter_by(phone=phone, status=1).first()
    if not tallyman:
        return BaseResponse(**PHONE_NOT_VALID_ERROR)

    if tallyman.check_password(password) != True:
        return BaseResponse(**PASSWORD_ERROR)

    tallyman.last_login = datetime.datetime.now()
    db.session.add(tallyman)

    login_log = TallymanLoginRecord()
    login_log.phone = tallyman.phone
    login_log.ip = cur_ip
    login_log.last_login = tallyman.last_login
    login_log.login_type = LOGIN_TYPE['password']
    login_log.created_at = datetime.datetime.now()
    login_log.updated_at = datetime.datetime.now()
    db.session.add(login_log)
    db.session.commit()

    new_token = generate_jwt(payload={"user_no": tallyman.user_no}, expiry=time.time() + 24 * 60 * 60)
    data['token'] = new_token
    data['user_name'] = tallyman.user_name
    data['phone'] = tallyman.phone
    data['level'] = tallyman.level

    return BaseResponse(data=data)


@tallyman_route.route('/machine_list', methods=['GET', 'POST'])
def get_tallyman_machine_list():
    machine_info = TallymanService.get_machine_list(g.user)

    return BaseResponse(data=machine_info)


@tallyman_route.route('/machine_info', methods=['GET', 'POST'])
def get_agent_module_list():
    json_data = request.get_json()
    machine_no = json_data["machine_no"]
    machine_info = TallymanService.get_machine_info(g.user, machine_no)

    return BaseResponse(data=machine_info)


@tallyman_route.route('hatch_list', methods=["post"])
def get_tallyman_hatch_list():
    json_data = request.get_json()
    machine_no = json_data["machine_no"]

    machine_info = TallymanService.get_machine_info(g.user, machine_no)

    if not machine_info:
        return jsonify(MACHINE_NOT_EXIST_ERROR)

    hatch_list = Hatch.query.filter_by(machine_no=machine_no).order_by(Hatch.hatch_no.asc()).all()
    if not hatch_list:
        return jsonify(HATCH_NOT_EXIST_ERROR)

    hatch_data = [{
        "machine_no": i.machine_no,
        "hatch_no": i.hatch_no,
        "production_id": i.production_id,
        "name": i.name,
        "status": i.status,
    } for i in hatch_list]

    return BaseResponse(data=hatch_data)


@tallyman_route.route("/tally_start", methods=["POST"])
def run_tally_start():
    json_data = request.get_json()
    machine_no = json_data['machine_no']
    tally_type = json_data["type"]  # 补货类型  1打开缺货锁  2打开指定仓号

    machine_info = TallymanService.get_machine_info(g.user, machine_no)

    if not machine_info:
        return jsonify(MACHINE_NOT_EXIST_ERROR)

    if tally_type == 1:
        hatch_list = Hatch.query.filter_by(machine_no=machine_no, status=2).order_by(Hatch.hatch_no.asc()).all()
        if not hatch_list:
            return jsonify(HATCH_NOT_EXIST_ERROR)

    elif tally_type == 2:
        hatch_nos = json_data['hatch_nos']
        hatch_list = Hatch.query.filter(Hatch.machine_no == machine_no, Hatch.status == 2,
                                        Hatch.hatch_no.in_(hatch_nos)).order_by(Hatch.hatch_no.asc()).all()
        if not hatch_list:
            return jsonify(HATCH_NOT_EXIST_ERROR)
    else:
        return BaseResponse()

    hatch_nos = []
    for i in hatch_list:
        hatch_nos.append(i.hatch_no)
        tally_record = TallyRecord()
        tally_record.user_no = g.user.user_no
        tally_record.user_name = g.user.user_name
        tally_record.machine_no = i.machine_no
        tally_record.hatch_no = i.hatch_no
        tally_record.production_id = i.production_id
        tally_record.production_name = i.name
        tally_record.status = 1
        db.session.add(tally_record)

    db.session.commit()

    return BaseResponse(data={"hatch_nos": hatch_nos})


@tallyman_route.route("/tally_over", methods=["POST"])
def run_tally_over():
    json_data = request.get_json()
    machine_no = json_data['machine_no']
    hatch_no = json_data['hatch_no']
    tally_type = json_data["type"]  # 补货类型  1全部缺货锁  2打开指定仓号
    user_id = g.user.id

    if tally_type == 1:
        for i in hatch_no:
            tally_record = TallyRecord.query.filter(TallyRecord.user_no == g.user.user_no,
                                                    TallyRecord.machine_no == machine_no,
                                                    TallyRecord.hatch_no == i,
                                                    TallyRecord.status == 1).order_by(TallyRecord.id.desc()).first()
            if tally_record:
                tally_record.status = 2
                db.session.add(tally_record)
                db.session.commit()
            hatch = Hatch.query.filter(Hatch.machine_no == machine_no, Hatch.hatch_no.in_(hatch_no),
                                       Hatch.status == 2).update({"status": 1})
            db.session.commit()

    elif tally_type == 2:
        tally_record = TallyRecord.query.filter(TallyRecord.user_no == g.user.user_no,
                                                TallyRecord.machine_no == machine_no,
                                                TallyRecord.hatch_no == hatch_no,
                                                TallyRecord.status == 1).order_by(TallyRecord.id.desc()).first()
        if tally_record:
            tally_record.status = 2
            db.session.add(tally_record)
        hatch = Hatch.query.filter_by(machine_no=machine_no, hatch_no=hatch_no, status=2).first()
        if hatch:
            hatch.status = 1
            db.session.add(hatch)
        db.session.commit()
    else:
        pass

    return BaseResponse()


@tallyman_route.route("/tally_report", methods=["POST"])
def get_tally_report():
    json_data = request.get_json()
    machine_no = json_data["machine_no"]
    machine_info = TallymanService.get_machine_info(g.user, machine_no)
    if not machine_info:
        return jsonify(MACHINE_NOT_EXIST_ERROR)
    empty_number = int(machine_info["empty_number"])

    over_number = TallyRecord.query.filter(TallyRecord.user_no == g.user.user_no,
                                           TallyRecord.machine_no == machine_no,
                                           TallyRecord.status == 2).count()

    return BaseResponse(data={"empty_number": empty_number, "over_number": over_number})


@tallyman_route.route('machine_activate', methods=["post"])
def run_machine_activate():
    json_data = request.get_json()
    qrcode_no = json_data["qrcode_no"]
    machine_no = json_data["machine_no"]
    mac = json_data["mac"]

    machine = Machine.query.filter_by(qrcode_no=qrcode_no, status=0).first()
    if machine and machine.machine_no == qrcode_no:
        machine.status = 1
        machine.mac = mac
        db.session.add(machine)
        db.session.commit()
        return BaseResponse()
    else:
        return jsonify(MACHINE_NOT_EXIST_ERROR)
