#!usr/bin/env python
# -*- coding:utf-8 _*-
"""
@version:
author:Aeolus
@file: hatch_portal.py
"""
import datetime
import json
import logging
import re
import time
from functools import reduce

from flask import Blueprint, request, jsonify, g
from sqlalchemy import func

from config.commen_config import USER_RENT_PREPAY_ID, RENT_SALE_LOCK
from config.wechat_config import platform_appid_config_list, pay_config_list
from models.base_model import db
from models.models import Machine, Hatch, Rent, WxUser, RentDetail
from service.rent_service import RentService
from service.wechat_service import WeChatPayService
from utils.error_code import Param_Invalid_Error, MACHINE_NOT_EXIST_ERROR, HATCH_NOT_EXIST_ERROR, \
    HATCH_NOT_ALL_EXIST_ERROR, WE_MINIAPP_PAY_FAIL, NO_RENT_RECORD, HATCH_COUNT_ERROR
from utils.my_redis_cache import redis_client
from utils.my_response import BaseResponse

logger = logging.getLogger(__name__)

rent_route = Blueprint('rent', __name__)


@rent_route.route("rent", methods=["POST"])
def create_rent():
    json_data = request.get_json()
    machine_no = json_data["machine_no"]
    productions = json_data["productions"]
    user = g.user

    # 验证机柜是否存在
    machine = Machine.query.filter_by(machine_no=machine_no).first()
    if not machine:
        return jsonify(MACHINE_NOT_EXIST_ERROR)

    total_fee = 0
    for id, count in productions.items():
        # 根据给的仓号去获取商品信息
        hatch_list = Hatch.query.filter(Hatch.machine_no == machine_no, Hatch.production_id == id,
                                        Hatch.status == 1).order_by(func.random()).all()

        if not hatch_list:
            return jsonify(HATCH_NOT_EXIST_ERROR)

        if len(hatch_list) < int(count):
            return jsonify(HATCH_COUNT_ERROR)

        total_fee += hatch_list[0].price * int(count)

    # 生成订单编号
    rent_no = RentService.create_order_no()

    # 配置微信订单数据
    wechat_service = WeChatPayService(app_id=platform_appid_config_list[user.platform],
                                      config_name=pay_config_list[machine.mch_platform])
    pay_data = {
        'body': '灰兔智能租借押金',  # 商品描述
        'out_trade_no': rent_no,  # 商户订单号
        'total_fee': total_fee,  # 总价
        'trade_type': "JSAPI",
        'openid': user.openid,
        'timeStamp': str(int(time.time())),
        'attach': {
            "productions": productions,
            "machine_no": machine_no,
            "user_id": user.id,
            "machine_id": machine.id,
            "platform": machine.mch_platform
        }
    }

    # logger.info(pay_data)
    # 微信下单接口
    pay_info = wechat_service.unifiedorder(pay_data)
    if not pay_info:
        return BaseResponse(**WE_MINIAPP_PAY_FAIL)

    # 微信下单成功，redis存prepay_id 5分钟失效
    redis_client.set(USER_RENT_PREPAY_ID + str(user.id) + rent_no, pay_info["prepay_id"], 300)

    return BaseResponse(data=pay_info)


@rent_route.route('/wx_pay_callback', methods=['GET', 'POST'])
def wx_pay_callback():
    """
    支付回调接口
    :return:
    """
    response_data = {
        'return_code': 'SUCCESS',
        'return_msg': 'SUCCESS'
    }
    error_data = {
        'return_code': 'FAIL',
        'return_msg': 'FAIL'
    }
    header = {'Content-Type': 'application/xml'}

    # 进行签名认证
    xml_data = request.data

    import xmltodict
    callback_data = dict(xmltodict.parse(xml_data)['xml'])
    logger.info(xml_data)
    logger.info(callback_data)
    attach = re.sub('\'', '\"', callback_data["attach"])
    print(attach)
    rent_data = json.loads(attach)
    platform = rent_data["platform"]
    machine_no = rent_data["machine_no"]
    machine_id = rent_data["machine_id"]
    productions = rent_data["productions"]
    user_id = rent_data["user_id"]
    platform = pay_config_list[platform]
    wechat_service = WeChatPayService(app_id=callback_data["appid"], config_name=platform)
    verify_result = wechat_service.verify_pay_callbak_sign(callback_data)
    if not verify_result:
        return xmltodict.unparse({'xml': response_data}, pretty=True), header

    # ================>  开始租借流程

    # 判断订单是否已经存在过
    rent_no = callback_data['out_trade_no']
    rent = Rent.query.filter_by(rent_no=rent_no).first()
    if rent:
        return xmltodict.unparse({'xml': error_data}, pretty=True), header

    machine = Machine.query.filter_by(machine_no=machine_no).first()

    try:
        lock_time = 0
        while lock_time < 3:
            # redis 加锁
            lock_res = redis_client.set(RENT_SALE_LOCK + str(machine.id),
                                        datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"), ex=60,
                                        nx=True)
            if lock_res:
                break
            lock_time += 1
            time.sleep(1)

        rent = Rent()
        rent.rent_no = rent_no
        rent.machine_no = machine.machine_no
        rent.user_id = rent_data["user_id"]
        rent.place_id = machine.place_id
        rent.total = int(callback_data["total_fee"])
        rent.add_time = datetime.datetime.now()
        rent.is_pay = 1
        rent.pay_time = datetime.datetime.strptime(callback_data["time_end"], "%Y%m%d%H%M%S")
        rent.rent_type = type
        rent.mch_platform = platform
        prepay_id = redis_client.get(USER_RENT_PREPAY_ID + str(rent_data["user_id"]) + rent_no)
        if prepay_id:
            rent.prepay_id = prepay_id

        total_fee = 0
        for id, count in productions.items():
            count = int(count)
            id = int(id)
            # 根据给的仓号去获取商品信息
            hatch_list = Hatch.query.filter(Hatch.machine_no == machine_no, Hatch.production_id == id,
                                            Hatch.status == 1).order_by(func.random()).all()

            if not hatch_list:
                return jsonify(HATCH_NOT_EXIST_ERROR)

            if len(hatch_list) < count:
                return jsonify(HATCH_NOT_ALL_EXIST_ERROR)
            hatchs = hatch_list[:count]
            for i in hatchs:
                if i.status == 1:
                    rent_detail = RentDetail()
                    rent_detail.rent_no = rent_no
                    rent_detail.user_id = rent.user_id
                    rent_detail.machine_no = rent.machine_no
                    rent_detail.hatch_no = i.hatch_no
                    rent_detail.production_id = i.production_id
                    rent_detail.name = i.name
                    rent_detail.title = i.title
                    rent_detail.brand_id = i.brand_id
                    rent_detail.brand_name = i.brand_name
                    rent_detail.cate_id = i.cate_id
                    rent_detail.cate_name = i.cate_name
                    rent_detail.price = i.price
                    rent_detail.img = i.img
                    rent_detail.tags = i.tags
                    rent_detail.content = i.content
                    rent_detail.summary = i.summary
                    i.status = 2
                    db.session.add(rent_detail)
                    db.session.add(i)

                    total_fee += i.price

        if total_fee != int(callback_data['total_fee']):
            return xmltodict.unparse({'xml': error_data}, pretty=True), header

        db.session.add(rent)
        db.session.commit()

    except Exception as e:
        redis_client.delete(RENT_SALE_LOCK + str(machine.id))
        return xmltodict.unparse({'xml': error_data}, pretty=True), header

    redis_client.delete(RENT_SALE_LOCK + str(machine.machine_id))
    return xmltodict.unparse({'xml': response_data}, pretty=True), header


@rent_route.route("/detail_info", methods=["POST"])
def get_rent_detail():
    json_data = request.get_json()
    rent_no = json_data["rent_no"]

    rent_info = Rent.query.filter_by(rent_no=rent_no).first()
    if not rent_info:
        return jsonify(NO_RENT_RECORD)

    rent_detail = RentDetail.query.filter_by(rent_no=rent_no, status=1).all()
    if not rent_detail:
        rent_detail = []

    return_data = {}
    return_data["rent_no"] = rent_info.rent_no
    return_data["machine_no"] = rent_info.machine_no
    return_data["user_id"] = rent_info.user_id
    return_data["place_id"] = rent_info.place_id
    return_data["total"] = rent_info.total
    return_data["real_total"] = rent_info.real_total
    return_data["add_time"] = rent_info.add_time
    return_data["pay_time"] = rent_info.pay_time

    detail_data = []
    for i in rent_detail:
        tmp_data = {}
        tmp_data["hatch_no"] = i.hatch_no
        tmp_data["production_id"] = i.production_id
        tmp_data["is_take"] = i.is_take
        tmp_data["name"] = i.name
        tmp_data["title"] = i.title
        tmp_data["brand_id"] = i.brand_id
        tmp_data["brand_name"] = i.brand_name
        tmp_data["cate_id"] = i.cate_id
        tmp_data["cate_name"] = i.cate_name
        tmp_data["price"] = i.price
        tmp_data["img"] = i.img
        tmp_data["tags"] = i.tags
        tmp_data["content"] = i.content
        tmp_data["summary"] = i.summary
        tmp_data["tags"] = i.tags
        detail_data.append(tmp_data)

    return_data["detail_data"] = detail_data

    return BaseResponse(data=return_data)


@rent_route.route("/detail_record", methods=["POST"])
def get_user_detail_record():
    json_data = request.get_json()
    page = int(json_data['page'])
    limit = int(json_data['limit'])
    user_id = g.user.id
    rent_detail = RentDetail.query.filter(RentDetail.user_id == user_id, RentDetail.status != -1).order_by(
        RentDetail.id.desc()).offset(
        (page - 1) * limit).limit(limit).all()
    if not rent_detail:
        rent_detail = []

    detail_data = []
    for i in rent_detail:
        tmp_data = {}
        tmp_data["machine_no"] = i.machine_no
        tmp_data["hatch_no"] = i.hatch_no
        tmp_data["production_id"] = i.production_id
        tmp_data["is_take"] = i.is_take
        tmp_data["name"] = i.name
        tmp_data["title"] = i.title
        tmp_data["brand_id"] = i.brand_id
        tmp_data["brand_name"] = i.brand_name
        tmp_data["cate_id"] = i.cate_id
        tmp_data["cate_name"] = i.cate_name
        tmp_data["price"] = i.price
        tmp_data["img"] = i.img
        tmp_data["tags"] = i.tags
        tmp_data["content"] = i.content
        tmp_data["summary"] = i.summary
        tmp_data["tags"] = i.tags
        tmp_data["created_at"] = i.created_at.strftime("%Y-%m-%d %H-%M-%S")
        detail_data.append(tmp_data)

    return BaseResponse(data=detail_data)


@rent_route.route("/rent_hatchs", methods=["POST"])
def get_user_rent_detail():
    json_data = request.get_json()
    rent_no = json_data['rent_no']
    user_id = g.user.id
    rent_detail = RentDetail.query.filter(RentDetail.user_id == user_id, RentDetail.rent_no == rent_no,
                                          RentDetail.status != -1).order_by(
        RentDetail.id.desc()).all()
    if not rent_detail:
        rent_detail = []

    detail_data = []
    for i in rent_detail:
        tmp_data = {}
        tmp_data["hatch_no"] = i.hatch_no
        tmp_data["production_id"] = i.production_id
        tmp_data["is_take"] = i.is_take
        tmp_data["name"] = i.name
        tmp_data["title"] = i.title
        tmp_data["price"] = i.price
        tmp_data["img"] = i.img
        tmp_data["status"] = i.status
        detail_data.append(tmp_data)

    return BaseResponse(data=detail_data)
