#! /usr/bin/.env python
from __future__ import print_function

import binascii
import logging

from smartcard.scard import *
import smartcard.util

logger = logging.getLogger(__name__)


class ScardService(object):

    def __init__(self, reader=None):
        if reader is None:
            self.reader = "ACS ACR1281 1S Dual Reader PICC 0"
        else:
            self.reader = reader

    def get_reader_state(self, state):
        reader, eventstate, atr = state
        logger.info(reader + " " + smartcard.util.toHexString(atr, smartcard.util.HEX))
        if eventstate & SCARD_STATE_EMPTY:
            message = 'Reader empty'
            logger.info(message)
            return 0, message
        if eventstate & SCARD_STATE_PRESENT:
            message = 'Card present in reader'
            logger.info(message)
            return 1, message
        if eventstate & SCARD_STATE_ATRMATCH:
            message = 'Card found'
            logger.info(message)
            return 2, message
        if eventstate & SCARD_STATE_UNAWARE:
            message = 'State unware'
            logger.info(message)
            return 3, message
        if eventstate & SCARD_STATE_IGNORE:
            message = 'Ignore reader'
            logger.info(message)
            return 4, message
        if eventstate & SCARD_STATE_UNAVAILABLE:
            message = 'Reader unavailable'
            logger.info(message)
            return 5, message
        if eventstate & SCARD_STATE_EXCLUSIVE:
            message = 'Card allocated for exclusive use by another application'
            logger.info(message)
            return 6, message
        if eventstate & SCARD_STATE_INUSE:
            message = 'Card in used by another application but can be shared'
            logger.info(message)
            return 7, message
        if eventstate & SCARD_STATE_MUTE:
            message = 'Card is mute'
            logger.info(message)
            return 8, message
        if eventstate & SCARD_STATE_CHANGED:
            message = 'State changed'
            logger.info(message)
            return 9, message
        if eventstate & SCARD_STATE_UNKNOWN:
            message = 'State unknowned'
            logger.info(message)
            return 10, message
        return False

    def establish_context(self):
        hresult, hcontext = SCardEstablishContext(SCARD_SCOPE_USER)
        if hresult != SCARD_S_SUCCESS:
            raise error(
                'Failed to establish context: ' + SCardGetErrorMessage(hresult))
        logger.info('Context established!')

        return hcontext

    def release_context(self, hcontext):
        hresult = SCardReleaseContext(hcontext)
        if hresult != SCARD_S_SUCCESS:
            raise error(
                'Failed to release context: ' + SCardGetErrorMessage(hresult))
        logger.info('Released context.')

    def find_self_reader(self, hcontext):
        hresult, readers = SCardListReaders(hcontext, [])
        if hresult != SCARD_S_SUCCESS:
            raise error(
                'Failed to list readers: ' + SCardGetErrorMessage(hresult))
        if self.reader not in readers:
            raise error('can not find reader ==> ' + self.reader)
        return True

    def handle_status_change(self, hcontext, readerstates):
        try:
            hresult, readerstates = SCardGetStatusChange(hcontext, INFINITE, readerstates)
            rs_code, message = self.get_reader_state(readerstates[0])
            if rs_code == 0:
                logger.info("请放入设备")
                return
            elif rs_code == 1:
                self.card_transmit(hcontext, self.reader)
            else:
                logger.info("未知错位，请重试")
                exit(1)
        except error as e:
            print(e)
        finally:
            hresult = SCardReleaseContext(hcontext)
            if hresult != SCARD_S_SUCCESS:
                raise error(
                    'Failed to release context: ' + SCardGetErrorMessage(hresult))
            print('Released context.')

    def my_scard_handler(self):
        try:
            hresult, hcontext = SCardEstablishContext(SCARD_SCOPE_USER)
            if hresult != SCARD_S_SUCCESS:
                raise error(
                    'Failed to establish context: ' + SCardGetErrorMessage(hresult))
            print('Context established!')

            try:
                # 获取读卡器列表，筛选PICC类型是否存在
                hresult, readers = SCardListReaders(hcontext, [])
                if hresult != SCARD_S_SUCCESS:
                    raise error(
                        'Failed to list readers: ' + SCardGetErrorMessage(hresult))
                if self.reader not in readers:
                    raise error('can not find reader ==> ' + self.reader)

                readerstates = [(self.reader, SCARD_STATE_UNAWARE)]

                while True:
                    hresult, readerstates = SCardGetStatusChange(hcontext, INFINITE, readerstates)
                    rs_code, message = self.get_reader_state(readerstates[0])
                    if rs_code == 0:
                        logger.info("请放入设备")
                        continue
                    elif rs_code == 1:
                        self.card_transmit(hcontext, self.reader)
                    else:
                        logger.info("未知错位，请重试")
                        exit(1)


            finally:
                hresult = SCardReleaseContext(hcontext)
                if hresult != SCARD_S_SUCCESS:
                    raise error(
                        'Failed to release context: ' + \
                        SCardGetErrorMessage(hresult))
                print('Released context.')

        except error as e:
            print(e)

    def card_transmit(self, hcontext, zreader, command):
        GET_RESPONSE = [0xA0, 0xC0, 0x00, 0x00]
        print('Trying to select DF_TELECOM of card in', zreader)

        try:
            hresult, hcard, dwActiveProtocol = SCardConnect(
                hcontext,
                zreader,
                SCARD_SHARE_SHARED, SCARD_PROTOCOL_T0 | SCARD_PROTOCOL_T1)
            if hresult != SCARD_S_SUCCESS:
                raise error(
                    'Unable to connect: ' +
                    SCardGetErrorMessage(hresult))
            print('Connected with active protocol', dwActiveProtocol)

            try:
                hresult, response = SCardTransmit(
                    hcard, dwActiveProtocol, command)
                if hresult != SCARD_S_SUCCESS:
                    raise error(
                        'Failed to transmit: ' +
                        SCardGetErrorMessage(hresult))
                    return False
                print('Selected DF_TELECOM: ' +
                      smartcard.util.toHexString(
                          response, smartcard.util.HEX))
                return response
            finally:
                hresult = SCardDisconnect(hcard, SCARD_UNPOWER_CARD)
                if hresult != SCARD_S_SUCCESS:
                    raise error(
                        'Failed to disconnect: ' +
                        SCardGetErrorMessage(hresult))
                print('Disconnected')

        except error as message:
            print(error, message)


if __name__ == '__main__':
    scard = ScardService()
    try:
        hcontext = scard.establish_context()
        result = scard.find_self_reader(hcontext)
        if not result:
            print('connect_result',
                  {'data': None, 'error_code': 1001, 'error_message': '读卡器连接失败, reader connect error'})
        # for i in range(5):
            # command = [0xFF, 0xB0, 0x00, 20, 0x10]
        # command = [0xFF, 0xD6, 0x00, 20, 0x10, 00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
        #            0x0C, 0x0D, 0x0E, 0x0F]
        # result = scard.card_transmit(hcontext, scard.reader, command)
        # if result:
        #     data = result[:16]
        #     return_code = result[-2:]
        # else:
        #     print("1002  reade_history_error")
    except error as e:
        logger.info(e)
    finally:
        hresult = scard.release_context(hcontext)
        logger.info('Released context.')
