<?php

namespace App\Http\Controllers\Api\Customers;


use App\Common\Enums\HtTab;
use App\Exceptions\Api\ApiException;
use App\Http\Controllers\LoginController;
use App\Modules\Enums\ErrorCode;
use App\Modules\Enums\LanguageType;
use App\Modules\Enums\SuccessCode;
use App\Modules\Models\Receive\Receive;
use App\Modules\Models\Share\Share;
use App\Repositories\Customers\RentRepository;
use App\Repositories\Customers\WeChatRepository;
use App\Repositories\Customers\WeGuideRepository;
use App\Repositories\Customers\WeGuideUseRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

/**
 * Class RentController
 * @package App\Http\Controllers\Api
 */
class WeGuideController extends LoginController
{
    /**
     * @var RentRepository
     */
    protected $weGuide;
    protected $weiChat;
    protected $weGuideUse;


    public function __construct(WeGuideRepository $weGuide, WeChatRepository $weChat, WeGuideUseRepository $weGuideUse)
    {
        parent::__construct();
        $this->weGuide = $weGuide;
        $this->weChat = $weChat;
        $this->weGuideUse = $weGuideUse;
    }

    function scan(Request $request)
    {
        Log::info("[weGuideScan]params:" . json_encode($request->all()));
        $guide_rule = $this->weGuide->getWeGuideRule($request->get('spot_code'));
        $is_own = $this->weGuide->isOwn($guide_rule['id'], $this->user->id);
        $response = [
            'guide_rule' => $this->weGuide->guideDataformat($guide_rule),
            'success_code' => SuccessCode::GUIDE_NEED_TO_BUY,
            'success_mes' => trans('api.success.guide_need_to_buy'),
        ];
        if ($is_own) {
            $response['success_code'] = SuccessCode::GUIDE_HAS_BUY;
            $response['success_mes'] = trans('api.success.guide_has_buy');
            $response['order_no'] = $is_own;

        }
        return $this->responseSuccess($response);
    }




    function MakeOrder(Request $request)
    {
        Log::info("[weGuideMakeOrder]params:" . json_encode($request->all()));
        $order = $this->weGuide->makeOrder($request->get('spot_code'), $this->user->id);
        $response = $this->weChat->submitOrderToWeChat(2,$order,$this->user->mini_program_open_id);

        return $this->responseSuccess($response);
    }

    function shareSpotInfo(Request $request)
    {
        Log::info("[weGuideShareSpotInfo]params:" . json_encode($request->all()));
        $spot_info = $this->weGuideUse->shareSpotInfo($request->get('order_no'));
        $response = [
            'spot_name' => $spot_info->spot->spotname,
            'spot_img' => HtTab::IMG_HEAD_URL.$spot_info->guideRule->guide_img,
            'spot_grade' => $spot_info->guideRule->level,
            'description' => $spot_info->guideRule->description,
        ];
        return $this->responseSuccess($response);
    }

    function getExplains(Request $request)
    {
        Log::info("[weGuideGetExplains]params:" . json_encode($request->all()));

        $share_info = $this->weGuide->getOrderShare($request->get('order_no'));
        //检测权限
        $own = $this->weGuide->isOwn($share_info->spot_id, $this->user->id);
        if ($own) {
            $explains = $this->weGuideUse->spot_explain($share_info, LanguageType::get($this->user->language));
        } else {
            throw new ApiException(ErrorCode::NO_GUIDE_PERMISSION, trans('api.error.no_guide_permission'));
        }
        return $this->responseSuccess($explains);
    }

    function shareHuman(Request $request)
    {
        Log::info("[weGuideShareHuman]params:" . json_encode($request->all()));
        $share_info = $this->weGuide->getOrderShare($request->get('order_no'));
        $res = $this->weGuide->getReceiveUser($share_info);
        return $this->responseSuccess($res);
    }


    function Receive(Request $request)
    {
        Log::info("[weGuideReceive]params:" . json_encode($request->all()));
        $response = [];
        $share = $this->weGuide->getOrderShare($request->get('order_no'));
        $receive_status = $this->weGuide->getReceiveStatus($share, $this->user->id);
        switch ($receive_status) {
            case 1 :
                $response['success_code'] = SuccessCode::GUIDE_HAS_RECEIVED_AVAILABLE;
                $response['success_mes'] = trans('api.success.guide_has_received_available');
                break;
            case 2:
                $response['success_code'] = SuccessCode::GUIDE_HAS_RECEIVED_EXPIRE;
                $response['success_mes'] = trans('api.success.guide_has_received_expire');
                break;
            case 3:
                $response['success_code'] = SuccessCode::GUIDE_HAS_RECEIVED_FULL;
                $response['success_mes'] = trans('api.success.guide_received_is_full');
                break;
            case 4:
                //领取分享
                $this->weGuide->setGuideReceive($share->id, $share->spot_id, $this->user->id,$share->order_no);
                $response['success_code'] = SuccessCode::GUIDE_RECEIVE_OK;
                $response['success_mes'] = trans('api.success.guide_receive_ok');
                break;
            case 5:
                //分享者拒绝领取
                $response['success_code'] = SuccessCode::GUIDE_SHARER;
                $response['success_mes'] = trans('api.success.guide_sharer');
                break;
            case 6:
                //已经享有服务无需领取(不是本次领取获得)
                $response['success_code'] = SuccessCode::GUIDE_HAD;
                $response['success_mes'] = trans('api.success.guide_had');
                break;
            default:
                break;

        }
        return $this->responseSuccess($response);

    }

    function record(Request $request){
        Log::info("[weGuideRecord]params:" . json_encode($request->all()));
        $data = $request->only('page', 'limit');
        $data['page'] = isset($data['page']) ? $data['page'] : 0;
        $data['limit'] = isset($data['limit']) ? $data['limit'] : HtTab::DEFAULT_LIMIT;
        $response = $this->weGuide->getMyRecord($this->user->id,$data['page'], $data['limit']);
        return $this->responseSuccess($response);
    }

    public function getQrcode(RentRepository $rentRepository)
    {
        $uid = $this->user->id;

        $rent_obj = $rentRepository->getLastOrder($uid);

        if($rent_obj && $rent_obj->spot && $rent_obj->spot->listen_img){
            $listen_img = $rent_obj->spot->listen_img;
            $response['data']['img'] = 'http://dev-1255927177.cossh.myqcloud.com'.$listen_img;
        }else{
            $response['data']['img'] = 'http://dev-1255927177.cossh.myqcloud.com/img/95aca070da59d427b7b82c44f1320dc0ef4bd051.png';
        }

        return $this->responseSuccess($response);
    }
}