<?php

namespace App\Repositories\Business;

use App\Exceptions\Api\ApiException;
use App\Modules\Enums\ErrorCode;
use App\Modules\Models\GuideRecord\GuideRecord;
use App\Modules\Models\Production\Production;
use App\Modules\Models\Rent\Rent;
use App\Modules\Models\RentTest\RentTest;
use App\Modules\Models\Yangzhou\Yangzhou;
use App\Modules\Repositories\BaseRepository;


/**
 * Class SpotRepository
 * @package App\Repositories\Spot
 */
class RentRepository extends BaseRepository
{

    public function getSpotRentDetail($rent_no, $spot_id)
    {

        $where = [
            ['rent_no', '=', $rent_no],
            ['spot_id', '=', $spot_id]
        ];
        $rent_obj = Rent::where($where)->with(['customer', 'spot','machine'])->first();
        if ($rent_obj == null) {
            throw new ApiException(ErrorCode::RENT_NOT_EXIST, trans('api.error.rent_not_exist'));
        }
        return $rent_obj;
    }

    public function getSpotRentDetailTest($rent_no, $spot_id)
    {

        $where = [
            ['rent_no', '=', $rent_no],
            ['spot_id', '=', $spot_id]
        ];
        $rent_obj = RentTest::where($where)->with(['customer', 'spot','machine'])->first();
        if ($rent_obj == null) {
            throw new ApiException(ErrorCode::RENT_NOT_EXIST, trans('api.error.rent_not_exist'));
        }
        return $rent_obj;
    }

    public function getSpotAllStatusRent($spot_id, $page, $limit, $search)
    {
        $where = [
            ['spot_id', '=', $spot_id],
            ['is_pay', '=', 1],
            ['is_cancel', '=', 0],
        ];
        if (isset($search['day'])) {
            $where[] = ['pay_time', '>=', $search['day'] . ' 00:00:00'];
            $where[] = ['pay_time', '<=', $search['day'] . ' 23:59:59'];
        }

        if (isset($search['total'])) {
            $where[] = ['real_total', '>', 0];
        }

        if (isset($search['month'])) {
            $start_time = $search['month'] . '-01 00:00:00';
            $end_time = date('Y-m', strtotime("+1 month {$search['month']}")) . '-01 00:00:00';
            $where[] = ['pay_time', '>=', $start_time];
            $where[] = ['pay_time', '<=', $end_time];
        }
        if (isset($search['status'])) {
            $where[] = ['is_over', '=', $search['status']];
        }

        return Rent::where($where)->orderBy('pay_time', 'desc')->skip($page * $limit)->take($limit)->get();
    }

    public function getSpotAllStatusRentTest($spot_id, $page, $limit, $search)
    {
        $where = [
            ['spot_id', '=', $spot_id],
            ['is_pay', '=', 1],
            ['is_cancel', '=', 0],
        ];
        if (isset($search['day'])) {
            $where[] = ['pay_time', '>=', $search['day'] . ' 00:00:00'];
            $where[] = ['pay_time', '<=', $search['day'] . ' 23:59:59'];
        }

        if (isset($search['total'])) {
            $where[] = ['real_total', '>', 0];
        }

        if (isset($search['month'])) {
            $start_time = $search['month'] . '-01 00:00:00';
            $end_time = date('Y-m', strtotime("+1 month {$search['month']}")) . '-01 00:00:00';
            $where[] = ['pay_time', '>=', $start_time];
            $where[] = ['pay_time', '<=', $end_time];
        }
        if (isset($search['status'])) {
            $where[] = ['is_over', '=', $search['status']];
        }

        return RentTest::where($where)->orderBy('pay_time', 'desc')->skip($page * $limit)->take($limit)->get();
    }


    public function getTodayRent($spot_id)
    {
        $start_time = date('Y-m-d') . ' 00:00:00';
        $end_time = date('Y-m-d') . ' 23:59:59';
        $where = [
            ['is_out', '=', 1],
            ['spot_id', '=', $spot_id],
            ['created_at', '>=', $start_time],
            ['created_at', '<=', $end_time]
        ];
        $rent_number = Production::where($where)->count();
        return $rent_number;
    }

    public function getTodayRentTest($spot_id)
    {
        $start_time = date('Y-m-d') . ' 00:00:00';
        $end_time = date('Y-m-d') . ' 23:59:59';
        $where = [
            ['is_pay', '=', 1],
            ['spot_id', '=', $spot_id],
            ['pay_time', '>=', $start_time],
            ['pay_time', '<=', $end_time],
            ['is_cancel', '=', 0]
        ];
//        $rent_number = Rent::where($where)->sum('is_cancel');
        $rent_number = RentTest::where($where)->count();
        return $rent_number;
    }

    public function getYangzhouData($spot_id)
    {
        $today = date('Y-m-d');
        $where = [
            ['cur_date', '=', $today]
        ];
        return Yangzhou::where($where)->get();
    }

    public function getTodayValidRent($spot_id)
    {
        $start_time = date('Y-m-d') . ' 00:00:00';
        $end_time = date('Y-m-d') . ' 23:59:59';
        $where = [
            ['is_out', '=', 1],
            ['is_return', '=', 1],
            ['total', '>', 0],
            ['spot_id', '=', $spot_id],
            ['created_at', '>=', $start_time],
            ['created_at', '<=', $end_time]
        ];
        $rent_number = Production::where($where)->count();
        return $rent_number;
    }

    public function getTodayValidRentTest($spot_id)
    {
        $start_time = date('Y-m-d') . ' 00:00:00';
        $end_time = date('Y-m-d') . ' 23:59:59';
        $where = [
            ['is_pay', '=', 1],
            ['spot_id', '=', $spot_id],
            ['pay_time', '>=', $start_time],
            ['pay_time', '<=', $end_time],
            ['is_cancel', '=', 0],
            ['real_total', '>', 0]
        ];
//        $rent_number = Rent::where($where)->sum('is_cancel');
        $rent_number = RentTest::where($where)->count();
        return $rent_number;
    }

    public function getDiscount($spot_id)
    {
        $where = [
            ['spot_id', '=', $spot_id],
        ];
        $discount_number = Rent::where($where)->has('consumeOrders')->count();
        return $discount_number;
    }

    public function rentFinish($spot_id)
    {
        $where = [
            ['is_over', '=', 1],
            ['spot_id', '=', $spot_id],
            ['is_cancel', '=', 0],
        ];
        $rent_number = Rent::where($where)->count();
        return $rent_number;
    }

    public function rentFinishTest($spot_id)
    {
        $where = [
            ['is_over', '=', 1],
            ['spot_id', '=', $spot_id],
            ['is_cancel', '=', 0],
        ];
        $rent_number = RentTest::where($where)->count();
        return $rent_number;
    }

    public function getTodayIncome($spot_id)
    {
        $start_time = date('Y-m-d') . ' 00:00:00';
        $end_time = date('Y-m-d') . ' 23:59:59';
        $where = [
            ['is_over', '=', 1],
            ['spot_id', '=', $spot_id],
            ['over_time', '>=', $start_time],
            ['over_time', '<=', $end_time],
            ['pay_time', '>=', $start_time],
            ['pay_time', '<=', $end_time],

        ];
        $rent_total = Rent::where($where)->sum('real_total');
        return $rent_total;
    }

    public function getTodayIncomeTest($spot_id)
    {
        $start_time = date('Y-m-d') . ' 00:00:00';
        $end_time = date('Y-m-d') . ' 23:59:59';
        $where = [
            ['is_over', '=', 1],
            ['spot_id', '=', $spot_id],
            ['over_time', '>=', $start_time],
            ['over_time', '<=', $end_time],
            ['pay_time', '>=', $start_time],
            ['pay_time', '<=', $end_time],

        ];
        $rent_total = RentTest::where($where)->sum('real_total');
        return $rent_total;
    }

    public function getThisMonthIncome($spot_id)
    {
        $time = date('Y-m', time());
        $start_time = $time . '-01 00:00:00';
        $end_time = date('Y-m-d H:i:s');
        $where = [
            ['is_over', '=', 1],
            ['spot_id', '=', $spot_id],
            ['over_time', '>=', $start_time],
            ['over_time', '<=', $end_time],

        ];
        $rent_total = Rent::where($where)->sum('real_total');
        return $rent_total;

    }

    public function getThisMonthIncomeTest($spot_id)
    {
        $time = date('Y-m', time());
        $start_time = $time . '-01 00:00:00';
        $end_time = date('Y-m-d H:i:s');
        $where = [
            ['is_over', '=', 1],
            ['spot_id', '=', $spot_id],
            ['over_time', '>=', $start_time],
            ['over_time', '<=', $end_time],

        ];
        $rent_total = RentTest::where($where)->sum('real_total');
        return $rent_total;

    }

    public function getSpotRentRecord($spot_id, $page, $limit, $search)
    {
        $where = [
            ['spot_id', '=', $spot_id],
            ['is_over', '=', 1]
        ];
        if (isset($search['day'])) {
            $where[] = ['over_time', '>=', $search['day'] . ' 00:00:00'];
            $where[] = ['over_time', '<=', $search['day'] . ' 23:59:59'];
        }

        if (isset($search['month'])) {
            $start_time = $search['month'] . '-01 00:00:00';
            $end_time = date('Y-m', strtotime("+1 month {$search['month']}")) . '-01 00:00:00';
            $where[] = ['over_time', '>=', $start_time];
            $where[] = ['over_time', '<=', $end_time];
        }

        return Rent::where($where)->orderBy('over_time', 'desc')->skip($page * $limit)->take($limit)->get();
    }

    public function getSpotRentRecordTest($spot_id, $page, $limit, $search)
    {
        $where = [
            ['spot_id', '=', $spot_id],
            ['is_over', '=', 1]
        ];
        if (isset($search['day'])) {
            $where[] = ['over_time', '>=', $search['day'] . ' 00:00:00'];
            $where[] = ['over_time', '<=', $search['day'] . ' 23:59:59'];
        }

        if (isset($search['month'])) {
            $start_time = $search['month'] . '-01 00:00:00';
            $end_time = date('Y-m', strtotime("+1 month {$search['month']}")) . '-01 00:00:00';
            $where[] = ['over_time', '>=', $start_time];
            $where[] = ['over_time', '<=', $end_time];
        }

        return RentTest::where($where)->orderBy('over_time', 'desc')->skip($page * $limit)->take($limit)->get();
    }


}
