# -*- coding: utf-8 -*-
import json
import time
import urllib.parse

import requests

from Config.common_config import ONENET_CONFIG as config, LIUYUAN_CONFIG


class OneNetApi(object):
    _key = None
    _base_url = None
    _raw_response = ''
    _http_code = 200
    _error_no = 0
    _error = ''
    _ALLOW_METHODS = ['GET', 'PUT', 'POST', 'DELETE']

    def __init__(self, key=None, base_url=None):
        self._key = key
        self._base_url = base_url

    def get_raw_response(self):
        return self._raw_response

    def get_error(self):
        return self._error

    def get_error_no(self):
        return self._error_no

    def get_http_code(self):
        return self._http_code

    def get_cur_api_key(self):
        return self._key

    def get_liuyuan_api_key(self):
        return LIUYUAN_CONFIG

    def device(self, id):
        if not id:
            return False
        api = '/devices/{}'.format(id)
        return self._call(api)

    def device_list(self, page=1, page_size=30, key_words=None, tag=None, is_online=None, is_private=None,
                    device_ids=None):
        params = {"page": 1, "page_size": 30}

        if isinstance(page, int):
            params["page"] = page if page else 1
        if isinstance(page_size, int):
            params["page_size"] = page_size if page_size else 30

        if key_words:
            params["key_words"] = key_words

        if tag:
            params["tag"] = tag

        if is_private:
            params["is_private"] = is_private

        if device_ids:
            params["device_ids"] = device_ids
        if is_online:
            params["is_online"] = "true"

        params_str = urllib.parse.urlencode(params)
        api = '/devices?' + params_str
        return self._call(api)

    def device_add(self, device):
        api = '/devices'
        return self._call(api, method="POST", data=device)

    def device_edit(self, id, device):
        if not id:
            return False
        api = '/devices/{}'.format(id)
        return self._call(api, "PUT", device)

    def device_delete(self, id):
        if not id:
            return False
        api = "/devices/{id}".format(id=id)
        return self._call(api, 'DELETE')

    def datastream(self, device_id, datastream_id):
        """
        数据流相关
        :param device_id: 设备id
        :param datastream_id:数据流id
        :return: False表示失败， json数据表示返回价格
        """
        if not device_id or not datastream_id:
            return False
        datastream_id = str(datastream_id).replace(" ", "%20")
        api = "/devices/{0}/datastreams/{1}".format(device_id, datastream_id)
        return self._call(api)

    def datastream_of_dev(self, device_id):
        """
        获取某个设备下面的数据流
        :param device_id:   设备id
        :return:
        """
        if not device_id:
            return False
        api = '/devices/{}/datastreams'.format(device_id)
        return self._call(api)

    def datastream_add(self, device_id, datastream):
        """
        新增数据流
        :param device_id:设备id
        :param datastream: 数据流
        :return:
        """
        if not device_id or not datastream:
            return False
        api = '/devices/{}/datastreams'.format(device_id)
        return self._call(api, "POST", datastream)

    def datastream_edit(self, device_id, datastream_id, datastream):
        if not device_id or datastream_id or datastream:
            return False

        api = '/devices/{0}/datastreams/{1}'.format(device_id, datastream_id)
        return self._call(api, "PUT", datastream)

    def datastream_delete(self, device_id, datasteam_id):
        if not device_id or not datasteam_id:
            return False
        api = '/devices/{}/datasteams/{}'.format(device_id, datasteam_id)
        return self._call(api, "DELETE")

    def datapoint_add(self, device_id, datastream_id, datas):
        """
        添加数据点
        :param device_id:
        :param datastream_id:
        :param datas:  {timestamp:data}
        :return:
        """
        if not datas:
            return True
        if not device_id or not datastream_id:
            return False

        datastream_data = []
        for k, v in datas.items():
            k = time.strftime('%Y-%m-%dT%H:%M:%S', time.localtime(k))
            datastream_data.append({
                "at": k,
                "value": v
            })
        api_data = {
            "datastreams": {
                "id": datastream_id,
                "datapoints": datastream_data
            }
        }

        api = "/devices/{device_id}/datapoints".format(device_id=device_id)
        return self._call(api, "POST", api_data)

    def datapoint_multi_add(self, device_id, datas):
        """
        多个datastream一次添加
        :param device_id:
        :param datas:   {'datastream_id':{'timestamp':data}}
        :return:
        """

        if not datas:
            return True
        if not device_id:
            return False
        api_data = {
            "datastreams": []
        }
        for datastream_id, value in datas.items():
            datastream_data = []
            for k, v in value.items():
                k = time.strftime('%Y-%m-%dT%H:%M:%S', time.localtime(k))
                datastream_data.append({
                    "at": k,
                    "value": v
                })
            if not datastream_data:
                continue
            api_data["datastreams"].append({
                "id": datastream_id,
                "datapoints": datastream_data
            })

        api = "/devices/{device_id}/datapoints".format(device_id=device_id)
        return self._call(api, "POST", api_data)

    def datapoint_get(self, device_id, datastream_id, start_time=None, end_time=None, limit=None, cursor=None):
        """
         2015-04-13 OneNet更新后，将不再支持以下参数:
         sort_time 可选，指定时按时间倒序排，最新时间在前面
         page 指定页码, 可选
         per_page 指定每页输出数据点个数,可选, 默认300，最多1000
         当前datastream更像是数据流的操作。废弃之前 datapoint_list, datapoint_multi_list 方法
        :param datastream_id:
        :param start_time:
        :param end_time:
        :param limit:
        :param cursor:
        :return:
        """
        if not device_id or datastream_id:
            return False
        return self.datapoint_multi_get(device_id, start_time, end_time, limit, cursor, datastream_id)

    def datapoint_multi_get(self, device_id, start_time=None, end_time=None, limit=None, cursor=None,
                            datastream_ids=None):
        # TODO  暂时用不到，等待开发
        if not device_id:
            return False

        params = {}

        if datastream_ids:
            if isinstance(datastream_ids, list):
                datastream_ids = ",".join(datastream_ids)
            params["datastream_id"] = datastream_ids

        pass

    def datapoint_delete(self, device_id, datastream_id, start_time=None, end_time=None):
        # TODO  暂时用不到，等待开发
        pass

    def datapoint_multi_delete(self, device_id, start_time=None, end_time=None):
        # TODO  暂时用不到，等待开发
        pass

    def trigger(self, trigger_id):
        # TODO  暂时用不到，等待开发
        pass

    def trigger_list(self, page=None, per_page=None, title=None):
        # TODO  暂时用不到，等待开发
        pass

    def trigger_add(self, trigger):
        # TODO  暂时用不到，等待开发
        pass

    def trigger_edit(self, trigger_id, trigger):
        # TODO  暂时用不到，等待开发
        pass

    def trigger_delete(self, trigger_id):
        # TODO  暂时用不到，等待开发
        pass

    def api_key(self, dev_id=None, key=None, page=None, per_page=None):
        # TODO  暂时用不到，等待开发
        pass

    def api_key_add(self, dev_id, title):
        # TODO  暂时用不到，等待开发
        pass

    def api_key_edit(self, key, title, resource=None):
        # TODO  暂时用不到，等待开发
        pass

    def api_key_delete(self, api_key):
        # TODO  暂时用不到，等待开发
        pass

    def request_log(self, device_id, start_time):
        # TODO  暂时用不到，等待开发
        pass

    def send_data_to_edp(self, device_id, qos, timeout, sms):
        if not device_id:
            return False

        api = "/cmds?device_id={device_id}&qos={qos}&timeout={timeout}".format(device_id=device_id, qos=qos,
                                                                               timeout=timeout)

        if device_id in ['83831673', '83831674', '83831675', '83831676']:
            return self._call_liuyuan(api, "POST", sms)

        return self._call(api, "POST", sms)

    def send_data_to_mqtt(self, topic, sms):
        # TODO  暂时用不到，等待开发
        pass

    def get_dev_status(self, cmd_uuid):
        # TODO  暂时用不到，等待开发
        pass

    def get_dev_status_resp(self, cmd_uuid):
        # TODO  暂时用不到，等待开发
        pass

    def _startendtime_to_startduration(self, start_time, end_time):
        # TODO  暂时用不到，等待开发
        pass

    def _call(self, url, method="GET", data=None, heasers=None):
        result = self._rawcall(url, method, data, heasers)
        if not result:
            return False

        return result

    def _rawcall(self, url, method, data=None, heasers=None):
        url = self._paddingUrl(url)

        self._error_no = 0
        self._error = None

        if not url:
            self._http_code = 500
            return False

        if method not in self._ALLOW_METHODS:
            self._http_code = 500
            return False

        if heasers:
            heasers = heasers

        kwargs = {"headers": {"api-key": self.get_cur_api_key()}}
        if data:
            if isinstance(data, list) or isinstance(data, dict):
                data = json.dumps(data).encode('utf-8')
            kwargs["data"] = data
        if heasers:
            kwargs["headers"].update(heasers)

        result = requests.request(method, url, **kwargs)
        print(result.text)

        if result:
            self._http_code = result.status_code
            result = json.loads(result.text)
            return result
        else:
            return {}

    def _call_liuyuan(self, url, method="GET", data=None, heasers=None):
        result = self._rawcall_liuyuan(url, method, data, heasers)
        if not result:
            return False

        return result

    def _rawcall_liuyuan(self, url, method, data=None, heasers=None):
        url = self._paddingUrl(url)

        self._error_no = 0
        self._error = None

        if not url:
            self._http_code = 500
            return False

        if method not in self._ALLOW_METHODS:
            self._http_code = 500
            return False

        if heasers:
            heasers = heasers

        kwargs = {"headers": {"api-key": self.get_liuyuan_api_key()}}
        if data:
            if isinstance(data, list) or isinstance(data, dict):
                data = json.dumps(data).encode('utf-8')
            kwargs["data"] = data
        if heasers:
            kwargs["headers"].update(heasers)

        result = requests.request(method, url, **kwargs)
        print(result.text)

        if result:
            self._http_code = result.status_code
            result = json.loads(result.text)
            return result
        else:
            return {}

    def _paddingUrl(self, url):
        if not url:
            return url

        if url[0] != '/':
            url = '/' + url
        return self._base_url + url

    def _beforeCall(self, ch, url, method, data):
        pass

    def _afterCall(self, ch, url, method, data, ret):
        pass

    def _afterDecode(self, url, method, data, ori_ret, ret):
        pass
