from flask import Flask, request, url_for, jsonify, g
from flask_cors import CORS

from Config.common_config import PLATFORM
from Libs.ErrorTips import TOKEN_NOT_PROVIDER_ERROR, TOKEN_NOT_VALID_ERROR, TOKEN_EXPIRE_ERROR
from Libs.Logger import logger
from Model.Agent.AgentAccountModel import AgentAccount
from Service.AgentService import AgentService
from Model.Base import db
from Redis_Cache import redis

from Controller.IndexController import route_index
from Controller.ToolController import route_tool
from Controller.AccountController import route_account
from Controller.ReportController import route_report
from Controller.RentController import route_rent
from Controller.RepairController import route_repair
from Controller.SpotController import route_spot
from Controller.InvoiceController import route_invoice

app = Flask(__name__)

app.config.from_object('Config.base_config')
CORS(app)

# 初始化数据库
db.init_app(app)
# 初始化redis
redis.init_app(app)

app.register_blueprint(route_index, url_prefix='/business/index')
app.register_blueprint(route_tool, url_prefix='/business/tool')
app.register_blueprint(route_account, url_prefix='/business/account')
app.register_blueprint(route_report, url_prefix='/business/report')
app.register_blueprint(route_rent, url_prefix='/business/rent')
app.register_blueprint(route_repair, url_prefix='/business/repair')
app.register_blueprint(route_spot, url_prefix='/business/spot')
app.register_blueprint(route_invoice, url_prefix='/business/invoice')


@app.before_request
def log_enter_interface():
    logger.info("#################### 进入 {} 接口 ####################".format(request.path))


@app.before_request
def get_platform():
    platform = request.headers.get('platform', None)
    g.platform = PLATFORM.get(platform, 0)


@app.before_request
def verify_auth_token():
    '''
    用户登录验证token
    :return:
    '''
    NO_AUTH_CHECK_URL = [url_for('account.test'), url_for('account.login'), url_for('account.send_code'),
                         url_for('tool.tool_reset')]
    if request.path not in NO_AUTH_CHECK_URL:
        token = request.headers.get('token')
        if not token:
            return jsonify(TOKEN_NOT_PROVIDER_ERROR)
        result = AgentService.check_agent_token(token)
        if result == 1:
            return jsonify(TOKEN_NOT_VALID_ERROR)
        elif result == 2:
            return jsonify(TOKEN_EXPIRE_ERROR)
        else:
            g.user = result
            logger.info("本次登录用户名称：{0}，手机号：{1}，类型：{2}".format(result.user_name, result.phone, result.level_desc))


@app.after_request
def log_out_interface(environ):
    logger.info("#################### 退出 {} 接口 ####################\n".format(request.path))
    return environ


if __name__ == '__main__':
    app.run(port='5056')
