# -*- coding: utf-8 -*-
import hashlib
import uuid
import xml.etree.ElementTree as ET

import requests

from Config.common_config import MINA_APP
from Libs.Logger import logger


class WeChatService():
    def __init__(self, app_id=None, app_key=None, mch_id=None, pay_key=None, pay_cert_path=None, pay_key_path=None):
        self.app_id = MINA_APP["app_id"] if not app_id else app_id
        self.app_key = MINA_APP["app_key"] if not app_key else app_key
        self.mch_id = MINA_APP["mch_id"] if not mch_id else mch_id
        self.pay_key = MINA_APP["pay_key"] if not pay_key else pay_key
        self.pay_cert_path = MINA_APP["wx_pay_cert_path"] if not pay_cert_path else pay_cert_path
        self.pay_key_path = MINA_APP["wx_pay_key_path"] if not pay_key_path else pay_key_path

        self.default_data = {
            "appid": self.app_id,
            "mch_id": self.mch_id,
            "nonce_str": self.get_nonce_str()
        }

    def create_sign(self, pay_data):
        '''
    	生成签名
    	:return:
    	'''

        stringA = '&'.join(["{0}={1}".format(k, pay_data.get(k)) for k in sorted(pay_data)])
        stringSignTemp = '{0}&key={1}'.format(stringA, self.pay_key)
        sign = hashlib.md5(stringSignTemp.encode("utf-8")).hexdigest()
        return sign.upper()

    def refund(self, refund_data=None):
        '''
        申请退款信息
        :param refund_data:
        :return:
        '''
        refund_data.update(self.default_data)
        sign = self.create_sign(refund_data)
        refund_data['sign'] = sign
        xml_data = self.dict_to_xml(refund_data)
        headers = {'Content-Type': 'application/xml'}
        url = "https://api.mch.weixin.qq.com/secapi/pay/refund"
        cert = (self.pay_cert_path, self.pay_key_path)
        r = requests.post(url=url, data=xml_data.encode('utf-8'), headers=headers, cert=cert)
        r.encoding = 'utf-8'
        logger.info(r.text)
        if r.status_code == 200:
            _return_data = self.xml_to_dict(r.text)
            if _return_data["return_code"] == "SUCCESS":
                if _return_data["result_code"] == "SUCCESS":
                    return True
        return False

    def query_refund_info(self, refund_data=None):
        '''
        查询退款信息
        :param refund_data:
        :return:
        '''
        refund_data.update(self.default_data)
        sign = self.create_sign(refund_data)
        refund_data['sign'] = sign
        xml_data = WeChatService.dict_to_xml(refund_data)
        headers = {'Content-Type': 'application/xml'}
        url = 'https://api.mch.weixin.qq.com/pay/refundquery'
        r = requests.post(url=url, data=xml_data.encode('utf-8'), headers=headers)
        r.encoding = 'utf-8'
        logger.info("查询退款信息参数")
        logger.info(r.text)
        if r.status_code == 200:
            _return_data = WeChatService.xml_to_dict(r.text)
            return _return_data
        return False

    @staticmethod
    def get_nonce_str():
        '''
        获取随机字符串
        :return:
        '''
        return str(uuid.uuid4()).replace('-', '')

    @staticmethod
    def dict_to_xml(dict_data):
        '''
        dict to xml
        :param dict_data:
        :return:
        '''
        xml = ["<xml>"]
        for k, v in dict_data.items():
            xml.append("<{0}>{1}</{0}>".format(k, v))
        xml.append("</xml>")
        return "".join(xml)

    @staticmethod
    def xml_to_dict(xml_data):
        '''
        xml to dict
        :param xml_data:
        :return:
        '''
        xml_dict = {}
        root = ET.fromstring(xml_data)
        for child in root:
            xml_dict[child.tag] = child.text
        return xml_dict
