import axios, { AxiosInstance, InternalAxiosRequestConfig, AxiosResponse } from 'axios'

import { showMessage } from './status'
import { Toast } from 'antd-mobile'
import type { ToastHandler } from 'antd-mobile/es/components/toast'

const axiosInstance: AxiosInstance = axios.create({
  baseURL: process.env.REACT_APP_BASE_URL,
  headers: {
    Accept: 'application/json',
    'Content-Type': 'application/json',
  },
})

let showToast: ToastHandler | null = null

// axios实例拦截响应
axiosInstance.interceptors.response.use(
  (response: AxiosResponse) => {
    showToast && showToast.close()
    if (response.status === 200) {
      if (response.data.data) return response.data.data
      return response.data
    } else {
      console.log(showMessage(response.status))
      return response
    }
  },
  // 请求失败
  (error: any) => {
    const { response } = error
    if (response) {
      // 请求已发出，但是不在2xx的范围
      Toast.show({
        icon: 'fail',
        content: showMessage(response.status),
      })

      return Promise.reject(response.data)
    } else {
      Toast.show({
        icon: 'fail',
        content: '网络连接异常,请稍后再试!',
      })
    }
  }
)

// axios实例拦截请求
axiosInstance.interceptors.request.use(
  (config: InternalAxiosRequestConfig) => {
    const token = localStorage.getItem('app_token')
    if (token) {
      config.headers.Authorization = `Bearer ${token}`
    }
    showToast = Toast.show({
      icon: 'loading',
      content: '加载中…',
      duration: 0,
    })
    return config
  },
  (error: any) => {
    return Promise.reject(error)
  }
)

export default axiosInstance
